/*
 * LaTeD Version 1.1
 * (c) Gene Ressler 1993, 94, 97
 *   de8827@trotter.usma.edu
 *
 * LaTeD is a graphical editor for drawings in the LaTeX "picture" 
 * environment.  It runs under MSDOS or in a Windows DOS box.  The
 * distribution includes full sources, including LaTeX source for 
 * its documentation.
 *
 * No warranty of this software is expressed or implied by the author.
 *
 * Copy and use this program freely for any purpose except for sale
 * of the program (including the source code) itself.  That is, 
 * no one can copy this program for the purpose of providing it to 
 * another person in exchange for money or other compensation, even 
 * if this program is only part of the exchange.
 *
 * All copies of computer source code in this distribution, whether
 * copies in whole or in part, must have this notice attached.
 */

/* CHECKS.C --- Check boxes of text entries. */

#include <stdlib.h>
#include <graphics.h>
#include <assert.h>
#include "window.h"
#include "key.h"

enum {

checks_border_width = 1,
checks_fg_color = cBLACK,
checks_border_color = checks_fg_color,
checks_bg_color = cWHITE,
checks_disabled_color = cLIGHTGRAY,
checks_x_margin = 4,
checks_box_size = 11,     /* Best as an odd number. */
checks_button_spacing = 12,
checks_button_button = bLEFT,
checks_focus_color = cLIGHTRED,

};

LOCAL(int) title_height(CHECKS ck)
{
  return ck->title == NULL ? 0 : textheight(ck->title);
}

LOCAL(void) X(int x, int y, int half_size)
{
  line(x - half_size, y - half_size, x + half_size, y + half_size);
  line(x - half_size, y + half_size, x + half_size, y - half_size);
}

LOCAL(void) draw_checks_entry(CHECKS_ENTRY ce)
{
  int half_size, center_y, box_color, text_color, X_color;
  CHECKS ck;

  ck = (CHECKS)ce->window.parent;

  assert(ce->pos == ce - ck->entries);
  assert(0 <= ck->focus && ck->focus < ck->n_entries);

  if (!w_status_p(&ck->window, wsVISIBLE))
    return;

  half_size = (checks_box_size - 1) / 2;
  center_y = ce->window.height / 2;

  push_graphics_state(&ce->window, 0);

  /* Figure out colors. */
  if (ck_status_p(ck, ckENABLED)) {
    X_color = (ck->on & bit(ce->pos)) ? checks_fg_color : checks_bg_color;
    box_color = ck_status_p(ck, ckHAVE_FOCUS) && (ce->pos == ck->focus) ? 
		  checks_focus_color : checks_fg_color;
    text_color = checks_fg_color;
  }
  else {
    X_color = checks_bg_color;
    box_color = text_color = checks_disabled_color;
  }
  setcolor(box_color);
  rectangle(0, center_y - half_size, 2 * half_size, center_y + half_size);
  setcolor(text_color);
  settextjustify(LEFT_TEXT, CENTER_TEXT);
  out_hot_textxy(checks_box_size + checks_x_margin, center_y, &ce->text);
  setcolor(X_color);
  X(half_size, center_y, half_size - 2);

  pop_graphics_state();
}

LOCAL(void) draw_checks_entry_protected(CHECKS_ENTRY ce)
{
  protect_cursor(&ce->window);
  draw_checks_entry(ce);
  unprotect_cursor();
}

LOCAL(void) draw_checks(CHECKS ck)
{
  int i;
  CHECKS_ENTRY ce;

  if (!w_status_p(&ck->window, wsVISIBLE))
    return;
  push_graphics_state(&ck->window, 0);
  setcolor(checks_fg_color);
  protect_cursor(&ck->window);
  if (ck->title != NULL) {
    settextjustify(LEFT_TEXT, TOP_TEXT);
    setfillstyle(SOLID_FILL, checks_bg_color);
    bar(checks_x_margin, 
	-checks_border_width, 
	3*checks_x_margin + textwidth(ck->title) - 1, textheight(ck->title) - 1);
    outtextxy(2*checks_x_margin, 0, ck->title);
  }
  for (i = 0, ce = ck->entries; i < ck->n_entries; ++i, ++ce)
    draw_checks_entry(ce);
  unprotect_cursor();
  pop_graphics_state();
}

LOCAL(void) handle_checks_map(EVENT e)
{
  draw_checks((CHECKS)e->map.window);
}

/* Toggle check if checks is enabled. */
LOCAL(int) toggle_check(CHECKS_ENTRY ce)
{
  int pos;
  CHECKS ck = (CHECKS)ce->window.parent;

  if (!ck_status_p(ck, rENABLED))
    return 0;

  pos = ce->pos;
  ck->on ^= bit(pos);
  draw_checks_entry_protected(ce);
  (*ck->action.code)((ck->on & bit(pos)) ? (0x8000 | pos) : pos, ck->action.env);
  return 1;
}

/* Move the check focus. */
LOCAL(void) move_focus(CHECKS ck, int new_focus)
{
  int old_focus, n;

  n = ck->n_entries;
  while (new_focus <  0) new_focus += n;
  while (new_focus >= n) new_focus -= n;
  old_focus = ck->focus;
  ck->focus = new_focus;
  protect_cursor(&ck->window);
  draw_checks_entry(&ck->entries[old_focus]);
  draw_checks_entry(&ck->entries[new_focus]);
  unprotect_cursor();
}

LOCAL(void) handle_entry_press(EVENT e)
{
  if (e->mouse.button != checks_button_button)
    return;
  toggle_check((CHECKS_ENTRY)e->mouse.window);
}

LOCAL(void) handle_entry_hotkey(EVENT e)
{
  CHECKS_ENTRY ce = (CHECKS_ENTRY)e->mouse.window;
  if (key_hot_p(e->hotkey.key, &ce->text))
    toggle_check(ce);
}

BeginDefDispatch(checks_entry)
  Dispatch(eBUTTON_PRESS, handle_entry_press)
  Dispatch(eVISIBLE_HOTKEY, handle_entry_hotkey)
EndDefDispatch(checks_entry)

LOCAL(void) handle_checks_gain_focus(EVENT e)
#define ck ((CHECKS)e->focus.window)
{
  ck->status |= bit(rHAVE_FOCUS);
  draw_checks_entry_protected(&ck->entries[ck->focus]);
}
#undef ck

LOCAL(void) handle_checks_lose_focus(EVENT e)
#define ck ((CHECKS)e->focus.window)
{
  ck->status &= notbit(rHAVE_FOCUS);
  draw_checks_entry_protected(&ck->entries[ck->focus]);
}
#undef ck

LOCAL(void) handle_checks_key(EVENT e)
#define ck ((CHECKS)e->keystroke.window)
{
  switch (e->keystroke.key) {

    case UP_ARROW:
      move_focus(ck, ck->focus - 1);
      break;

    case DOWN_ARROW:
      move_focus(ck, ck->focus + 1);
      break;

    case HOME:
      move_focus(ck, 0);
      break;

    case END:
      move_focus(ck, ck->n_entries - 1);
      break;

    case '\r':
    case ' ':
      toggle_check(&ck->entries[ck->focus]);
      break;

    case '\t':
      unset_focus(e->keystroke.window);
      break;

    default:
      refuse_keystroke(e);
      break;
  }
}
#undef ck

BeginDefDispatch(checks)
  Dispatch(eMAP, handle_checks_map)
  Dispatch(eGAIN_FOCUS, handle_checks_gain_focus)
  Dispatch(eLOSE_FOCUS, handle_checks_lose_focus)
  Dispatch(eKEYSTROKE, handle_checks_key)
EndDefDispatch(checks)

void open_checks(CHECKS ck, int x, int y, WINDOW parent)
{
  int i, width, height, yb;
  CHECKS_ENTRY p;

  width = 0;
  for (i = 0, p = ck->entries; i < ck->n_entries; ++i, ++p) 
    width = max(width, textwidth(p->text.str));
  /* margin button margin text margin */
  width += 3*checks_x_margin + checks_box_size;
  height = title_height(ck) + (ck->n_entries + 1) * checks_button_spacing;
  open_window(&ck->window, parent, x, y, width, height, 
	      checks_border_width, checks_border_color, checks_bg_color, 
	      bit(eMAP)|bit(eGAIN_FOCUS)|bit(eLOSE_FOCUS)|bit(eKEYSTROKE));
  SetDispatch(&ck->window, checks);
  for (i = 0, p = ck->entries, yb = title_height(ck) + checks_button_spacing; 
       i < ck->n_entries; 
       ++i, ++p, yb += checks_button_spacing) {
    open_window(&p->window, &ck->window, 
		checks_x_margin, yb - (checks_box_size - 1)/2, 
		width - 2 * checks_x_margin, checks_box_size,
		0, TRANSPARENT, TRANSPARENT, 
		bit(eBUTTON_PRESS)|bit(eVISIBLE_HOTKEY));
    SetDispatch(&p->window, checks_entry);
    p->pos = i;
    map_window(&p->window);
  }
  ck->focus = 0;
}

#ifdef NOT_USED 

void enable_checks(CHECKS ck)
{
  ck->status |= bit(rENABLED);
  draw_checks(ck);
}

void disable_checks(CHECKS ck)
{
  ck->status &= notbit(rENABLED);
  draw_checks(ck);
}

#endif

void set_checks_on_mask(CHECKS ck, unsigned long on)
{
  ck->on = on;
  draw_checks(ck);
}
