@q Copyright 2012-2020 Alexander Shibakov@>
@q Copyright 2002-2014 Free Software Foundation, Inc.@>
@q This file is part of SPLinT@>

@q SPLinT is free software: you can redistribute it and/or modify@>
@q it under the terms of the GNU General Public License as published by@>
@q the Free Software Foundation, either version 3 of the License, or@>
@q (at your option) any later version.@>

@q SPLinT is distributed in the hope that it will be useful,@>
@q but WITHOUT ANY WARRANTY; without even the implied warranty of@>
@q MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the@>
@q GNU General Public License for more details.@>

@q You should have received a copy of the GNU General Public License@>
@q along with SPLinT.  If not, see <http://www.gnu.org/licenses/>.@>

@** The lexer.
\ifbootstrapmode
    \input ldman.sty
    \modebootstrap
    \input dcols.sty
    \setupfootnotes
    \def\MRI{} 
    \def\ld{}
\fi
The lexer used by \ld\ is almost straightforward. There are a few
facilities (\Cee\ header files, some output functions) needed by the
lexer that are conviniently coded into the \Cee\ code run by the
driver routines that make the lexer more complex than it should have
been but the function of each such facility can be easily clarified
using this documentation and occasionally referring to the manual for
the \bison\ parser which is part of this distribution.
@(ldl.ll@>=
@G
 @> @<\ld\ lexer definitions@> @= 
%{@> @<\ld\ lexer \Cee\ preamble@> @=%}
 @> @<\ld\ lexer options@> @= 
%%
 @> @<\ld\ token regular expressions@> @= 
%%
@O
void define_all_states( void ) {
  @<Collect state definitions for the \ld\ lexer@>@;
}
@o
@g

@ @<\ld\ lexer options@>=
@G
%option bison-bridge
%option noyywrap nounput noinput reentrant 
%option noyy_top_state
%option debug
%option stack
%option outfile="ldl.c"
@g

@ @<\ld\ lexer \Cee\ preamble@>=
#include <stdint.h>
#include <stdbool.h>

@ @<Collect state definitions for the \ld\ lexer@>=
#define _register_name( name ) @[Define_State( #name, name )@]
#include "ldl_states.h"
#undef _register_name

@ The character classes used by the scanner as well as
lexer state declarations have been put in the definitions section of
the input file. No attempt has been made to clean up the definitions
of the character classes. 
@<\ld\ lexer definitions@>=
@<\ld\ lexer states@>@;
@G(fs1)
CMDFILENAMECHAR  [_a-zA-Z0-9\/\.\\_\+\$\:\[\]\\\,\=\&\!\<\>\-\~]
CMDFILENAMECHAR1 [_a-zA-Z0-9\/\.\\_\+\$\:\[\]\\\,\=\&\!\<\>\~]
FILENAMECHAR1    [_a-zA-Z\/\.\\\$\_\~]
SYMBOLCHARN      [_a-zA-Z\/\.\\\$\_\~0-9]
FILENAMECHAR     [_a-zA-Z0-9\/\.\-\_\+\=\$\:\[\]\\\,\~]
WILDCHAR         [_a-zA-Z0-9\/\.\-\_\+\=\$\:\[\]\\\,\~\?\*\^\!]
WHITE            [ \t\n\r]+
NOCFILENAMECHAR  [_a-zA-Z0-9\/\.\-\_\+\$\:\[\]\\\~]
V_TAG            [.$_a-zA-Z][._a-zA-Z0-9]*
V_IDENTIFIER     [*?.$_a-zA-Z\[\]\-\!\^\\]([*?.$_a-zA-Z0-9\[\]\-\!\^\\]|::)*
@g

@ The lexer uses different sets of rules depending on the context and the current state.
These can be changed from the lexer itself or externally by the parser
(as is the case in \ld\
implementation). \locallink{stateswitchers}Later\endlink, a number of
helper macros implement state switching so that the state names are
very rarely used explicitly. Keeping all the state declarations in the
same section simplifies the job of the
\locallink{bootstrapstates}bootstrap parser\endlink, as well.
\ifbootstrapmode\immediate\openout\stlist=ldl_states.h\fi
@<\ld\ lexer states@>=
@G
%s SCRIPT
%s EXPRESSION
%s BOTH
%s DEFSYMEXP
%s MRI
%s VERS_START
%s VERS_SCRIPT
%s VERS_NODE
@g

@*1 Macros for lexer functions.
The \locallink{pingpong}state switching\endlink\ `ping-pong' between the lexer and the parser aside,
the \ld\ lexer is very traditional. One implementation choice
deserving some attention is the treatment of comments. The
difficulty of implementing \Cee\ style comment scanning using regular
expressions is well-known so an often used alternative is a
special function that simply skips to the end of the comment. This is
exactly what the \ld\ lexer does with an aptly named |comment()|
function. The typesetting parser uses the \.{\\ldcomment} macro for
the same purpose. For the curious, here is a \flex\ style regular
expression defining \Cee\ comments\footnote{Taken from W.~McKeeman's site
at
\url{http://www.cs.dartmouth.edu/~mckeeman/cs118/assignments/comment.html}
and adapted to \flex\ syntax.}:
$$
\hbox{\.{"/*" ("/"\yl[\^*/]\yl"*"+[\^*/])* "*"+ "/"}}
$$
This expression does not handle {\it every\/} practical situation,
however, since it assumes that the end of line character can be
matched like any other. Neither does it detect some often made
mistakes such as attempting to nest comments. A few minor
modifications can fix this deficiency, as well as add some error
handling, however, for the sake of consistency, the approach taken
here mirrors the one in the original \ld.

The top level of the \.{\\ldcomment} macro simply bypasses the state
setup of the lexer and enters a `|while| loop' in the input
routine. This macro is a reasonable approximation of the functionality
provided by |comment()|.
@<Additional macros for the \ld\ lexer/parser@>=
@G(t)
\def\ldcomment{%
    \let\oldyyreturn\yyreturn
    \let\oldyylextail\yylextail
    \let\yylextail\yymatch        %/* start inputting characters until {\tt *}{\tt /} is seen */
    \let\yyreturn\ldcommentskipchars
}
@g

@ The rest of the |while| loop merely waits for the \.{*/} combination.
@<Additional macros for the \ld\ lexer/parser@>=
@G(t)
\def\ldcommentskipchars{%
    \ifnum\yycp@@=`*
        \yybreak{\let\yyreturn\ldcommentseekslash\yyinput}%
                                  %/* {\tt *} found, look for {\tt /} */
    \else
        \yybreak{\yyinput}%       %/* keep skipping characters */
    \yycontinue
}%

\def\ldcommentseekslash{%
    \ifnum\yycp@@=`/
        \yybreak{\ldcommentfinish}%/* {\tt /} found, exit */
    \else
        \ifnum\yycp@@=`*
            \yybreak@@{\yyinput}%  %/* keep skipping {\tt *}'s looking for a {\tt /} */
        \else
            \yybreak@@{\let\yyreturn\ldcommentskipchars\yyinput}%  
                                  %/* found a character other than {\tt *} or {\tt /} */
        \fi
    \yycontinue
}%
@g

@ Once the end of the comment has been found, resume lexing the input
stream.
@<Additional macros for the \ld\ lexer/parser@>=
@G(t)
\def\ldcommentfinish{%
    \let\yyreturn\oldyyreturn
    \let\yylextail\oldyylextail
    \yylextail
}
@g

@ The semantics of the macros defined above do not quite match that
of the |comment()| function. The most significant difference is that
the portion of the action following \.{\\ldcomment} expands {\it
before\/} the comment characters are skipped. In most applications,
|comment()| is the last function called so this would not limit the use
of \.{\\ldcomment} too dramatically.

A more intuitive and easier to use version of \.{\\ldcomment} is
possible, however, if \.{\\yylextail} is not used inside actions (in the case of
an `optimized' lexer the restriction is even weaker, namely,
\.{\\yylextail} merely has to be absent in the portion of the action
following \.{\\ldcomment}).
@<Additional macros for the \ld\ lexer/parser@>=
@G(t)
\def\ldcomment#1\yylextail{%
    \let\oldyyreturn\yyreturn
    \def\yylexcontinuation{#1\yylextail}%
    \let\yyreturn\ldcommentskipchars %/* start inputting characters until {\tt *}{\tt /} is seen */
    \yymatch
}

\def\ldcommentfinish{%
    \let\yyreturn\oldyyreturn
    \yylexcontinuation
}
@g

@ \namedspot{pretendbufferswlex}The same idea can be applied to 
`\locallink{pretendbuffersw}pretend buffer switching\endlink'. Whenever
the `real' \ld\ parser encounters an \prodstyle{INCLUDE} command, it
switches the input buffer for the lexer and waits for the lexer to
return the tokens from the file it just opened. When the lexer scans
the end of the included file, it returns a special token, \prodstyle{END} that
completes the appropriate production and lets the parser continue with
its job. 

We would like to simulate the file inclusion by inserting the
appropriate end of file marker for the lexer (a double
\.{\\yyeof}). After the relevant production completes, the marker
has to be cleaned up from the input stream (the lexer is designed to
leave it intact). The macros below are designed to handle this assignment.  
@<Additional macros for the \ld\ lexer/parser@>=
@G(t)
\def\ldcleanyyeof#1\yylextail{%
    \let\oldyyinput\yyinput
    \def\yyinput\yyeof\yyeof{\let\yyinput\oldyyinput#1\yylextail}%
    \yymatch
}
@g

@*1 Regular expressions.
The `heart' of any lexer is the collection of regular expressions that
describe the {\it tokens\/} of the appropriate language. The variey of
tokens recognized by \ld\ is quite extensive and is described in the
sections that follow.

Variable names and algebraic operations come first.
@<\ld\ token regular expressions@>=
@G(fs2)
<BOTH,SCRIPT,EXPRESSION,VERS_START,VERS_NODE,VERS_SCRIPT>"/*" {@> @[TeX_( "/ldcomment/yylexnext" );@]@=}
<DEFSYMEXP>"-"                                   {@> @[TeX_( "/yylexreturnchar" );@]@=}
<DEFSYMEXP>"+"                                   {@> @[TeX_( "/yylexreturnchar" );@]@=}
<DEFSYMEXP>{FILENAMECHAR1}{SYMBOLCHARN}*         {@> @[TeX_( "/yylexreturnsym{NAME}" );@]@=}
<DEFSYMEXP>"="                                   {@> @[TeX_( "/yylexreturnchar" );@]@=}
<MRI,EXPRESSION>"$"([0-9A-Fa-f])+                {@> @<Return an absolute hex constant@> @=}
<MRI,EXPRESSION>([0-9A-Fa-f])+(H|h|X|x|B|b|O|o|D|d) {@> @<Return a constant in a specific radix@>@=}
<SCRIPT,DEFSYMEXP,MRI,BOTH,EXPRESSION>((("$"|0[xX])([0-9A-Fa-f])+)|(([0-9])+))(M|K|m|k)? {
                                                  @> @<Return a constant with a multiplier@>@=}
<BOTH,SCRIPT,EXPRESSION,MRI>"]"                  {@> @[TeX_( "/yylexreturnchar" );@]@=}
<BOTH,SCRIPT,EXPRESSION,MRI>"["                  {@> @[TeX_( "/yylexreturnchar" );@]@=}
<BOTH,SCRIPT,EXPRESSION,MRI>"<<="                {@> @[TeX_( "/yylexreturnptr{LSHIFTEQ}" );@]@=}
<BOTH,SCRIPT,EXPRESSION,MRI>">>="                {@> @[TeX_( "/yylexreturnptr{RSHIFTEQ}" );@]@=}
<BOTH,SCRIPT,EXPRESSION,MRI>"||"                 {@> @[TeX_( "/yylexreturnptr{OROR}" );@]@=}
<BOTH,SCRIPT,EXPRESSION,MRI>"=="                 {@> @[TeX_( "/yylexreturnptr{EQ}" );@]@=}
<BOTH,SCRIPT,EXPRESSION,MRI>"!="                 {@> @[TeX_( "/yylexreturnptr{NE}" );@]@=}
<BOTH,SCRIPT,EXPRESSION,MRI>">="                 {@> @[TeX_( "/yylexreturnptr{GE}" );@]@=}
<BOTH,SCRIPT,EXPRESSION,MRI>"<="                 {@> @[TeX_( "/yylexreturnptr{LE}" );@]@=}
<BOTH,SCRIPT,EXPRESSION,MRI>"<<"                 {@> @[TeX_( "/yylexreturnptr{LSHIFT}" );@]@=}
<BOTH,SCRIPT,EXPRESSION,MRI>">>"                 {@> @[TeX_( "/yylexreturnptr{RSHIFT}" );@]@=}
<BOTH,SCRIPT,EXPRESSION,MRI>"+="                 {@> @[TeX_( "/yylexreturnptr{PLUSEQ}" );@]@=}
<BOTH,SCRIPT,EXPRESSION,MRI>"-="                 {@> @[TeX_( "/yylexreturnptr{MINUSEQ}" );@]@=}
<BOTH,SCRIPT,EXPRESSION,MRI>"*="                 {@> @[TeX_( "/yylexreturnptr{MULTEQ}" );@]@=}
<BOTH,SCRIPT,EXPRESSION,MRI>"/="                 {@> @[TeX_( "/yylexreturnptr{DIVEQ}" );@]@=}
<BOTH,SCRIPT,EXPRESSION,MRI>"&="                 {@> @[TeX_( "/yylexreturnptr{ANDEQ}" );@]@=}
<BOTH,SCRIPT,EXPRESSION,MRI>"|="                 {@> @[TeX_( "/yylexreturnptr{OREQ}" );@]@=}
<BOTH,SCRIPT,EXPRESSION,MRI>"&&"                 {@> @[TeX_( "/yylexreturnptr{ANDAND}" );@]@=}
<BOTH,SCRIPT,EXPRESSION,MRI>">"                  {@> @[TeX_( "/yylexreturnchar" );@]@=}
<BOTH,SCRIPT,EXPRESSION,MRI>","                  {@> @[TeX_( "/yylexreturnchar" );@]@=}
<BOTH,SCRIPT,EXPRESSION,MRI>"&"                  {@> @[TeX_( "/yylexreturnchar" );@]@=}
<BOTH,SCRIPT,EXPRESSION,MRI>"|"                  {@> @[TeX_( "/yylexreturnchar" );@]@=}
<BOTH,SCRIPT,EXPRESSION,MRI>"~"                  {@> @[TeX_( "/yylexreturnchar" );@]@=}
<BOTH,SCRIPT,EXPRESSION,MRI>"!"                  {@> @[TeX_( "/yylexreturnchar" );@]@=}
<BOTH,SCRIPT,EXPRESSION,MRI>"?"                  {@> @[TeX_( "/yylexreturnchar" );@]@=}
<BOTH,SCRIPT,EXPRESSION,MRI>"*"                  {@> @[TeX_( "/yylexreturnchar" );@]@=}
<BOTH,SCRIPT,EXPRESSION,MRI>"+"                  {@> @[TeX_( "/yylexreturnchar" );@]@=}
<BOTH,SCRIPT,EXPRESSION,MRI>"-"                  {@> @[TeX_( "/yylexreturnchar" );@]@=}
<BOTH,SCRIPT,EXPRESSION,MRI>"/"                  {@> @[TeX_( "/yylexreturnchar" );@]@=}
<BOTH,SCRIPT,EXPRESSION,MRI>"%"                  {@> @[TeX_( "/yylexreturnchar" );@]@=}
<BOTH,SCRIPT,EXPRESSION,MRI>"<"                  {@> @[TeX_( "/yylexreturnchar" );@]@=}
<BOTH,SCRIPT,EXPRESSION,MRI>"="                  {@> @[TeX_( "/yylexreturnchar" );@]@=}
<BOTH,SCRIPT,EXPRESSION,MRI>"}"                  {@> @[TeX_( "/yylexreturnchar" );@]@=}
<BOTH,SCRIPT,EXPRESSION,MRI>"{"                  {@> @[TeX_( "/yylexreturnchar" );@]@=}
<BOTH,SCRIPT,EXPRESSION,MRI>")"                  {@> @[TeX_( "/yylexreturnchar" );@]@=}
<BOTH,SCRIPT,EXPRESSION,MRI>"("                  {@> @[TeX_( "/yylexreturnchar" );@]@=}
<BOTH,SCRIPT,EXPRESSION,MRI>":"                  {@> @[TeX_( "/yylexreturnchar" );@]@=}
<BOTH,SCRIPT,EXPRESSION,MRI>";"                  {@> @[TeX_( "/yylexreturnchar" );@]@=}
@g

@ The bulk of tokens produced by the lexer are the keywords used
inside script files. File name syntax is listed as well, along with
miscellanea such as whitespace and version symbols.
@<\ld\ token regular expressions@>=
@G(fs2)
<BOTH,SCRIPT>"MEMORY"                            {@> @[TeX_( "/yylexreturnptr{MEMORY}" );@]@=}
<BOTH,SCRIPT>"REGION_ALIAS"                      {@> @[TeX_( "/yylexreturnptr{REGION_ALIAS}" );@]@=}
<BOTH,SCRIPT>"LD_FEATURE"                        {@> @[TeX_( "/yylexreturnptr{LD_FEATURE}" );@]@=}
<BOTH,SCRIPT,EXPRESSION>"ORIGIN"                 {@> @[TeX_( "/yylexreturnptr{ORIGIN}" );@]@=}
<BOTH,SCRIPT>"VERSION"                           {@> @[TeX_( "/yylexreturnptr{VERSIONK}" );@]@=}
<EXPRESSION,BOTH,SCRIPT>"BLOCK"                  {@> @[TeX_( "/yylexreturnptr{BLOCK}" );@]@=}
<EXPRESSION,BOTH,SCRIPT>"BIND"                   {@> @[TeX_( "/yylexreturnptr{BIND}" );@]@=}
<BOTH,SCRIPT,EXPRESSION>"LENGTH"                 {@> @[TeX_( "/yylexreturnptr{LENGTH}" );@]@=}
<EXPRESSION,BOTH,SCRIPT>"ALIGN"                  {@> @[TeX_( "/yylexreturnptr{ALIGN_K}" );@]@=}
<EXPRESSION,BOTH,SCRIPT>"DATA_SEGMENT_ALIGN"     {@> @[TeX_( "/yylexreturnptr{DATA_SEGMENT_ALIGN}" );@]@=}
<EXPRESSION,BOTH,SCRIPT>"DATA_SEGMENT_RELRO_END" {@> @[TeX_( "/yylexreturnptr{DATA_SEGMENT_RELRO_END}" );@]@=}
<EXPRESSION,BOTH,SCRIPT>"DATA_SEGMENT_END"       {@> @[TeX_( "/yylexreturnptr{DATA_SEGMENT_END}" );@]@=}
<EXPRESSION,BOTH,SCRIPT>"ADDR"                   {@> @[TeX_( "/yylexreturnptr{ADDR}" );@]@=}
<EXPRESSION,BOTH,SCRIPT>"LOADADDR"               {@> @[TeX_( "/yylexreturnptr{LOADADDR}" );@]@=}
<EXPRESSION,BOTH,SCRIPT>"ALIGNOF"                {@> @[TeX_( "/yylexreturnptr{ALIGNOF}" );@]@=}
<EXPRESSION,BOTH>"MAX"                           {@> @[TeX_( "/yylexreturnptr{MAX_K}" );@]@=}
<EXPRESSION,BOTH>"MIN"                           {@> @[TeX_( "/yylexreturnptr{MIN_K}" );@]@=}
<EXPRESSION,BOTH>"LOG2CEIL"                      {@> @[TeX_( "/yylexreturnptr{LOG2CEIL}" );@]@=}
<EXPRESSION,BOTH,SCRIPT>"ASSERT"                 {@> @[TeX_( "/yylexreturnptr{ASSERT_K}" );@]@=}
<BOTH,SCRIPT>"ENTRY"                             {@> @[TeX_( "/yylexreturnptr{ENTRY}" );@]@=}
<BOTH,SCRIPT,MRI>"EXTERN"                        {@> @[TeX_( "/yylexreturnptr{EXTERN}" );@]@=}
<EXPRESSION,BOTH,SCRIPT>"NEXT"                   {@> @[TeX_( "/yylexreturnptr{NEXT}" );@]@=}
<EXPRESSION,BOTH,SCRIPT>"sizeof_headers"         {@> @[TeX_( "/yylexreturnptr{SIZEOF_HEADERS}" );@]@=}
<EXPRESSION,BOTH,SCRIPT>"SIZEOF_HEADERS"         {@> @[TeX_( "/yylexreturnptr{SIZEOF_HEADERS}" );@]@=}
<EXPRESSION,BOTH,SCRIPT>"SEGMENT_START"          {@> @[TeX_( "/yylexreturnptr{SEGMENT_START}" );@]@=}
<BOTH,SCRIPT>"MAP"                               {@> @[TeX_( "/yylexreturnptr{MAP}" );@]@=}
<EXPRESSION,BOTH,SCRIPT>"SIZEOF"                 {@> @[TeX_( "/yylexreturnptr{SIZEOF}" );@]@=}
<BOTH,SCRIPT>"TARGET"                            {@> @[TeX_( "/yylexreturnptr{TARGET_K}" );@]@=}
<BOTH,SCRIPT>"SEARCH_DIR"                        {@> @[TeX_( "/yylexreturnptr{SEARCH_DIR}" );@]@=}
<BOTH,SCRIPT>"OUTPUT"                            {@> @[TeX_( "/yylexreturnptr{OUTPUT}" );@]@=}
<BOTH,SCRIPT>"INPUT"                             {@> @[TeX_( "/yylexreturnptr{INPUT}" );@]@=}
<EXPRESSION,BOTH,SCRIPT>"GROUP"                  {@> @[TeX_( "/yylexreturnptr{GROUP}" );@]@=}
<EXPRESSION,BOTH,SCRIPT>"AS_NEEDED"              {@> @[TeX_( "/yylexreturnptr{AS_NEEDED}" );@]@=}
<EXPRESSION,BOTH,SCRIPT>"DEFINED"                {@> @[TeX_( "/yylexreturnptr{DEFINED}" );@]@=}
<BOTH,SCRIPT>"CREATE_OBJECT_SYMBOLS"             {@> @[TeX_( "/yylexreturnptr{CREATE_OBJECT_SYMBOLS}" );@]@=}
<BOTH,SCRIPT>"CONSTRUCTORS"                      {@> @[TeX_( "/yylexreturnptr{CONSTRUCTORS}" );@]@=}
<BOTH,SCRIPT>"FORCE_COMMON_ALLOCATION"           {@> @[TeX_( "/yylexreturnptr{FORCE_COMMON_ALLOCATION}" );@]@=}
<BOTH,SCRIPT>"INHIBIT_COMMON_ALLOCATION"         {@> @[TeX_( "/yylexreturnptr{INHIBIT_COMMON_ALLOCATION}" );@]@=}
<BOTH,SCRIPT>"SECTIONS"                          {@> @[TeX_( "/yylexreturnptr{SECTIONS}" );@]@=}
<BOTH,SCRIPT>"INSERT"                            {@> @[TeX_( "/yylexreturnptr{INSERT_K}" );@]@=}
<BOTH,SCRIPT>"AFTER"                             {@> @[TeX_( "/yylexreturnptr{AFTER}" );@]@=}
<BOTH,SCRIPT>"BEFORE"                            {@> @[TeX_( "/yylexreturnptr{BEFORE}" );@]@=}
<BOTH,SCRIPT>"FILL"                              {@> @[TeX_( "/yylexreturnptr{FILL}" );@]@=}
<BOTH,SCRIPT>"STARTUP"                           {@> @[TeX_( "/yylexreturnptr{STARTUP}" );@]@=}
<BOTH,SCRIPT>"OUTPUT_FORMAT"                     {@> @[TeX_( "/yylexreturnptr{OUTPUT_FORMAT}" );@]@=}
<BOTH,SCRIPT>"OUTPUT_ARCH"                       {@> @[TeX_( "/yylexreturnptr{OUTPUT_ARCH}" );@]@=}
<BOTH,SCRIPT>"HLL"                               {@> @[TeX_( "/yylexreturnptr{HLL}" );@]@=}
<BOTH,SCRIPT>"SYSLIB"                            {@> @[TeX_( "/yylexreturnptr{SYSLIB}" );@]@=}
<BOTH,SCRIPT>"FLOAT"                             {@> @[TeX_( "/yylexreturnptr{FLOAT}" );@]@=}
<BOTH,SCRIPT>"QUAD"                              {@> @[TeX_( "/yylexreturnptr{QUAD}" );@]@=}
<BOTH,SCRIPT>"SQUAD"                             {@> @[TeX_( "/yylexreturnptr{SQUAD}" );@]@=}
<BOTH,SCRIPT>"LONG"                              {@> @[TeX_( "/yylexreturnptr{LONG}" );@]@=}
<BOTH,SCRIPT>"SHORT"                             {@> @[TeX_( "/yylexreturnptr{SHORT}" );@]@=}
<BOTH,SCRIPT>"BYTE"                              {@> @[TeX_( "/yylexreturnptr{BYTE}" );@]@=}
<BOTH,SCRIPT>"NOFLOAT"                           {@> @[TeX_( "/yylexreturnptr{NOFLOAT}" );@]@=}
<EXPRESSION,BOTH,SCRIPT>"NOCROSSREFS"            {@> @[TeX_( "/yylexreturnptr{NOCROSSREFS}" );@]@=}
<BOTH,SCRIPT>"OVERLAY"                           {@> @[TeX_( "/yylexreturnptr{OVERLAY}" );@]@=}
<BOTH,SCRIPT>"SORT_BY_NAME"                      {@> @[TeX_( "/yylexreturnptr{SORT_BY_NAME}" );@]@=}
<BOTH,SCRIPT>"SORT_BY_ALIGNMENT"                 {@> @[TeX_( "/yylexreturnptr{SORT_BY_ALIGNMENT}" );@]@=}
<BOTH,SCRIPT>"SORT"                              {@> @[TeX_( "/yylexreturnptr{SORT_BY_NAME}" );@]@=}
<BOTH,SCRIPT>"SORT_BY_INIT_PRIORITY"             {@> @[TeX_( "/yylexreturnptr{SORT_BY_INIT_PRIORITY}" );@]@=}
<BOTH,SCRIPT>"SORT_NONE"                         {@> @[TeX_( "/yylexreturnptr{SORT_NONE}" );@]@=}
<EXPRESSION,BOTH,SCRIPT>"NOLOAD"                 {@> @[TeX_( "/yylexreturnptr{NOLOAD}" );@]@=}
<EXPRESSION,BOTH,SCRIPT>"DSECT"                  {@> @[TeX_( "/yylexreturnptr{DSECT}" );@]@=}
<EXPRESSION,BOTH,SCRIPT>"COPY"                   {@> @[TeX_( "/yylexreturnptr{COPY}" );@]@=}
<EXPRESSION,BOTH,SCRIPT>"INFO"                   {@> @[TeX_( "/yylexreturnptr{INFO}" );@]@=}
<EXPRESSION,BOTH,SCRIPT>"OVERLAY"                {@> @[TeX_( "/yylexreturnptr{OVERLAY}" );@]@=}
<EXPRESSION,BOTH,SCRIPT>"ONLY_IF_RO"             {@> @[TeX_( "/yylexreturnptr{ONLY_IF_RO}" );@]@=}
<EXPRESSION,BOTH,SCRIPT>"ONLY_IF_RW"             {@> @[TeX_( "/yylexreturnptr{ONLY_IF_RW}" );@]@=}
<EXPRESSION,BOTH,SCRIPT>"SPECIAL"                {@> @[TeX_( "/yylexreturnptr{SPECIAL}" );@]@=}
<BOTH,SCRIPT>"o"                                 {@> @[TeX_( "/yylexreturnptr{ORIGIN}" );@]@=}
<BOTH,SCRIPT>"org"                               {@> @[TeX_( "/yylexreturnptr{ORIGIN}" );@]@=}
<BOTH,SCRIPT>"l"                                 {@> @[TeX_( "/yylexreturnptr{LENGTH}" );@]@=}
<BOTH,SCRIPT>"len"                               {@> @[TeX_( "/yylexreturnptr{LENGTH}" );@]@=}
<EXPRESSION,BOTH,SCRIPT>"INPUT_SECTION_FLAGS"    {@> @[TeX_( "/yylexreturnptr{INPUT_SECTION_FLAGS}" );@]@=}
<EXPRESSION,BOTH,SCRIPT>"INCLUDE"                {@> @[TeX_( "/yylexreturnptr{INCLUDE}" );@]@=}
<BOTH,SCRIPT>"PHDRS"                             {@> @[TeX_( "/yylexreturnptr{PHDRS}" );@]@=}
<EXPRESSION,BOTH,SCRIPT>"AT"                     {@> @[TeX_( "/yylexreturnptr{AT}" );@]@=}
<EXPRESSION,BOTH,SCRIPT>"ALIGN_WITH_INPUT"       {@> @[TeX_( "/yylexreturnptr{ALIGN_WITH_INPUT}" );@]@=}
<EXPRESSION,BOTH,SCRIPT>"SUBALIGN"               {@> @[TeX_( "/yylexreturnptr{SUBALIGN}" );@]@=}
<EXPRESSION,BOTH,SCRIPT>"HIDDEN"                 {@> @[TeX_( "/yylexreturnptr{HIDDEN}" );@]@=}
<EXPRESSION,BOTH,SCRIPT>"PROVIDE"                {@> @[TeX_( "/yylexreturnptr{PROVIDE}" );@]@=}
<EXPRESSION,BOTH,SCRIPT>"PROVIDE_HIDDEN"         {@> @[TeX_( "/yylexreturnptr{PROVIDE_HIDDEN}" );@]@=}
<EXPRESSION,BOTH,SCRIPT>"KEEP"                   {@> @[TeX_( "/yylexreturnptr{KEEP}" );@]@=}
<EXPRESSION,BOTH,SCRIPT>"EXCLUDE_FILE"           {@> @[TeX_( "/yylexreturnptr{EXCLUDE_FILE}" );@]@=}
<EXPRESSION,BOTH,SCRIPT>"CONSTANT"               {@> @[TeX_( "/yylexreturnptr{CONSTANT}" );@]@=}
<MRI>"#".*\n?                                    {@> @[TeX_( "/yylexnext" );@]@=}
<MRI>"\n"                                        {@> @[TeX_( "/yylexreturnptr{NEWLINE}" );@]@=}
<MRI>"*".*                                       {@> @[TeX_( "/yylexnext" );@]@=}
<MRI>";".*                                       {@> @[TeX_( "/yylexnext" );@]@=}
<MRI>"END"                                       {@> @[TeX_( "/yylexreturnptr{ENDWORD}" );@]@=}
<MRI>"ALIGNMOD"                                  {@> @[TeX_( "/yylexreturnptr{ALIGNMOD}" );@]@=}
<MRI>"ALIGN"                                     {@> @[TeX_( "/yylexreturnptr{ALIGN_K}" );@]@=}
<MRI>"CHIP"                                      {@> @[TeX_( "/yylexreturnptr{CHIP}" );@]@=}
<MRI>"BASE"                                      {@> @[TeX_( "/yylexreturnptr{BASE}" );@]@=}
<MRI>"ALIAS"                                     {@> @[TeX_( "/yylexreturnptr{ALIAS}" );@]@=}
<MRI>"TRUNCATE"                                  {@> @[TeX_( "/yylexreturnptr{TRUNCATE}" );@]@=}
<MRI>"LOAD"                                      {@> @[TeX_( "/yylexreturnptr{LOAD}" );@]@=}
<MRI>"PUBLIC"                                    {@> @[TeX_( "/yylexreturnptr{PUBLIC}" );@]@=}
<MRI>"ORDER"                                     {@> @[TeX_( "/yylexreturnptr{ORDER}" );@]@=}
<MRI>"NAME"                                      {@> @[TeX_( "/yylexreturnptr{NAMEWORD}" );@]@=}
<MRI>"FORMAT"                                    {@> @[TeX_( "/yylexreturnptr{FORMAT}" );@]@=}
<MRI>"CASE"                                      {@> @[TeX_( "/yylexreturnptr{CASE}" );@]@=}
<MRI>"START"                                     {@> @[TeX_( "/yylexreturnptr{START}" );@]@=}
<MRI>"LIST".*                                    {@> @[TeX_( "/yylexreturnptr{LIST}" );@]@=}
<MRI>"SECT"                                      {@> @[TeX_( "/yylexreturnptr{SECT}" );@]@=}
<EXPRESSION,BOTH,SCRIPT,MRI>"ABSOLUTE"           {@> @[TeX_( "/yylexreturnptr{ABSOLUTE}" );@]@=}
<MRI>"end"                                       {@> @[TeX_( "/yylexreturnptr{ENDWORD}" );@]@=}
<MRI>"alignmod"                                  {@> @[TeX_( "/yylexreturnptr{ALIGNMOD}" );@]@=}
<MRI>"align"                                     {@> @[TeX_( "/yylexreturnptr{ALIGN_K}" );@]@=}
<MRI>"chip"                                      {@> @[TeX_( "/yylexreturnptr{CHIP}" );@]@=}
<MRI>"base"                                      {@> @[TeX_( "/yylexreturnptr{BASE}" );@]@=}
<MRI>"alias"                                     {@> @[TeX_( "/yylexreturnptr{ALIAS}" );@]@=}
<MRI>"truncate"                                  {@> @[TeX_( "/yylexreturnptr{TRUNCATE}" );@]@=}
<MRI>"load"                                      {@> @[TeX_( "/yylexreturnptr{LOAD}" );@]@=}
<MRI>"public"                                    {@> @[TeX_( "/yylexreturnptr{PUBLIC}" );@]@=}
<MRI>"order"                                     {@> @[TeX_( "/yylexreturnptr{ORDER}" );@]@=}
<MRI>"name"                                      {@> @[TeX_( "/yylexreturnptr{NAMEWORD}" );@]@=}
<MRI>"format"                                    {@> @[TeX_( "/yylexreturnptr{FORMAT}" );@]@=}
<MRI>"case"                                      {@> @[TeX_( "/yylexreturnptr{CASE}" );@]@=}
<MRI>"extern"                                    {@> @[TeX_( "/yylexreturnptr{EXTERN}" );@]@=}
<MRI>"start"                                     {@> @[TeX_( "/yylexreturnptr{START}" );@]@=}
<MRI>"list".*                                    {@> @[TeX_( "/yylexreturnptr{LIST}" );@]@=}
<MRI>"sect"                                      {@> @[TeX_( "/yylexreturnptr{SECT}" );@]@=}
<EXPRESSION,BOTH,SCRIPT,MRI>"absolute"           {@> @[TeX_( "/yylexreturnptr{ABSOLUTE}" );@]@=}
<MRI>{FILENAMECHAR1}{NOCFILENAMECHAR}*           {@> @[TeX_( "/yylexreturnsym{NAME}" );@]@=}
<BOTH>{FILENAMECHAR1}{FILENAMECHAR}*             {@> @[TeX_( "/yylexreturnsym{NAME}" );@]@=}
<BOTH>"-l"{FILENAMECHAR}+                        {@> @[TeX_( "/yylexreturnsym{NAME}" );@]@=}
<EXPRESSION>{FILENAMECHAR1}{NOCFILENAMECHAR}*    {@> @[TeX_( "/yylexreturnsym{NAME}" );@]@=}
<EXPRESSION>"-l"{NOCFILENAMECHAR}+               {@> @[TeX_( "/yylexreturnsym{NAME}" );@]@=}
<SCRIPT>{WILDCHAR}*                              {@> @[@<Skip a possible comment and return a \prodstyle{NAME}@>@]@=}
<EXPRESSION,BOTH,SCRIPT,VERS_NODE>"\""[^\"]*"\"" {@> @[@<Return the \prodstyle{NAME} inside quotes@>@]@=}
<BOTH,SCRIPT,EXPRESSION>"\n"                     {@> @[TeX_( "/yylexnext" );@]@=}
<MRI,BOTH,SCRIPT,EXPRESSION>[ \t\r]+             {@> @[TeX_( "/yylexnext" );@]@=}
<VERS_NODE,VERS_SCRIPT>[:,;]                     {@> @[TeX_( "/yylexreturnchar" );@]@=}
<VERS_NODE>global                                {@> @[TeX_( "/yylexreturnptr{GLOBAL}" );@]@=}
<VERS_NODE>local                                 {@> @[TeX_( "/yylexreturnptr{LOCAL}" );@]@=}
<VERS_NODE>extern                                {@> @[TeX_( "/yylexreturnptr{EXTERN}" );@]@=}
<VERS_NODE>{V_IDENTIFIER}                        {@> @[TeX_( "/yylexreturnval{VERS_IDENTIFIER}" );@]@=}
<VERS_SCRIPT>{V_TAG}                             {@> @[TeX_( "/yylexreturnval{VERS_TAG}" );@]@=}
<VERS_START>"{"                                  {@> @[TeX_( "/yyBEGIN{VERS_SCRIPT}/yylexreturnchar" );@]@=}
@g

@ There is a bit of a trick to returning an absolute hex value. The
macros are looking for a \.{\$} suffix while the contents of
\.{\\yytext} start with \.{\\\$}.
@<Return an absolute hex constant@>=
@[TeX_( "/edef/next{/yylval{/nx/hexint{$/expandafter/eatone/the/yytext}" );@]@;
@[TeX_( "{/the/yyfmark}{/the/yysmark}}}/next" );@]@;
@[TeX_( "/yylexreturn{INT}" );@]@;

@ @<Return a constant in a specific radix@>=
@[TeX_( "/edef/next{/yylval{/nx/bint{/the/yytext}" );@]@;
@[TeX_( "{/the/yyfmark}{/the/yysmark}}}/next" );@]@;
@[TeX_( "/yylexreturn{INT}" );@]@;

@ @<Return a constant with a multiplier@>=
@[TeX_( "/edef/next{/yylval{/nx/anint{/the/yytext}" );@]@;
@[TeX_( "{/the/yyfmark}{/the/yysmark}}}/next" );@]@;
@[TeX_( "/yylexreturn{INT}" );@]@;

@ @<Additional macros for the \ld\ lexer/parser@>=
@G(t)
@=\def\matchcomment@@#1/*#2\yyeof#3#4{%@>@;
    \yystringempty{#1}{#3}{#4}%
}
\def\matchcomment#1{%
@=    \expandafter\matchcomment@@\the#1/*\yyeof@>@;
}
\def\ldstripquotes@@"#1"\yyeof{#1}
\def\ldstripquotes#1{%
    \yytext\expandafter\expandafter\expandafter
           {\expandafter\ldstripquotes@@\the\yytext\yyeof}%
    \yytextpure\expandafter\expandafter\expandafter
           {\expandafter\ldstripquotes@@\the\yytextpure\yyeof}%
}
@g

@ {\it Annoyingly, this pattern can match comments, and we have
longest match issues to consider.  So if the first two
characters are a comment opening, put the input back and
try again.}
@<Skip a possible comment and return a \prodstyle{NAME}@>=
  @[TeX_( "/matchcomment/yytextpure" );@]@;
  @[TeX_( "    {/yyless/tw@@/ldcomment}" );@]/*matched the beginning of a comment*/@;
  @[TeX_( "    {/yylexreturnsym{NAME}}" );@]@;

@ {\it No matter the state, quotes give what's inside.}
@<Return the \prodstyle{NAME} inside quotes@>=
  @[TeX_( "/ldstripquotes/yylexreturnsym{NAME}" );@]@;

@ @<Additional macros for the \ld\ lexer/parser@>=
@G(t)
\newcount\versnodenesting
\newcount\includestackptr
@g

@ Some syntax specific to version scripts.
@<\ld\ token regular expressions@>=
@G(fs2)
<VERS_SCRIPT>"{"                                 {@> @[TeX_( "/yyBEGIN{VERS_NODE}/versnodenesting=/z@@/yylexreturnchar" );@]@=}
<VERS_SCRIPT>"}"                                 {@> @[TeX_( "/yylexreturnchar" );@]@=}
<VERS_NODE>"{"                                   {@> @[TeX_( "/advance/versnodenesting/@@ne /yylexreturnchar" );@]@=}
<VERS_NODE>"}"                                   {@> @[TeX_( "/advance/versnodenesting/m@@ne" );@]@= 
                                                  @> @[TeX_( "/ifnum/versnodenesting</z@@" );@]@= 
                                                  @> @[TeX_( "    /yyBEGIN{VERS_SCRIPT}" );@]@= 
                                                  @> @[TeX_( "/fi" );@]@= 
                                                  @> @[TeX_( "/yylexreturnchar" );@]@=}
<VERS_START,VERS_NODE,VERS_SCRIPT>[\n]           {@> @[TeX_( "/yylexnext" );@]@=}
<VERS_START,VERS_NODE,VERS_SCRIPT>#.*            {@> @[TeX_( "/yylexnext" );@]@=}
<VERS_START,VERS_NODE,VERS_SCRIPT>[ \t\r]+       {@> @[TeX_( "/yylexnext" );@]@=}

<<EOF>>                                          {@> @[@<Process the end of (possibly included) file@>@]@=}

<SCRIPT,MRI,VERS_START,VERS_SCRIPT,VERS_NODE>.   {@> @[TeX_( "/yyfatal{bad character `/the/yytext' in script}" );@]@=}
<EXPRESSION,DEFSYMEXP,BOTH>.                     {@> @[TeX_( "/yyfatal{bad character `/the/yytext' in expression}" );@]@=} 
@g

@ @<Process the end of (possibly included) file@>=
  @[TeX_( "/advance/includestackptr/m@@ne" );@]@;
  @[TeX_( "/ifnum/includestackptr=/z@@" );@]@;
  @[TeX_( "    /yybreak{/yyterminate}" );@]@;
  @[TeX_( "/else" );@]@;
  @[TeX_( "    /yybreak{/ldcleanyyeof/yylexreturn{END}}" );@]@;
  @[TeX_( "/yycontinue" );@]@;

@*1 Parser-lexer interaction support. 
\namedspot{stateswitchers}Here are the long promised auxiliary
macros for switching lexer states and handling file input.
@<Additional macros for the \ld\ lexer/parser@>=
@G(t)
\def\ldlex@@script{\yypushstate{SCRIPT}}
\def\ldlex@@mri@@script{\yypushstate{MRI}}
\def\ldlex@@version@@script{\yypushstate{VERS_START}}
\def\ldlex@@version@@file{\yypushstate{VERS_SCRIPT}}
\def\ldlex@@defsym{\yypushstate{DEFSYMEXP}}
\def\ldlex@@expression{\yypushstate{EXPRESSION}}
\def\ldlex@@both{\yypushstate{BOTH}}
\let\ldlex@@popstate\yypopstate

\def\ldfile@@open@@command@@file#1{%
    \advance\includestackptr\@@ne
    \appendlnx\yytext@@seen{\yyeof\yyeof}%
    \yytextbackuptrue
}

\def\ldlex@@filename{}
@g

